# File Manifest Generator

A comprehensive Node.js utility that scans your website's file structure and generates a detailed JSON manifest for the static file browser system.

## 🚀 Quick Start

```bash
# Generate manifest with default settings
node generate-manifest.js

# Generate with verbose output
node generate-manifest.js --verbose

# Custom output file and path
node generate-manifest.js --path ./content --output my-manifest.json

# Using npm scripts
npm run generate
npm run generate:verbose
```

## 📋 Features

### Core Functionality
- **Recursive Directory Scanning**: Processes all sections (Section 01-09) and nested folders
- **File Metadata Collection**: Size, type, modification date, MIME type
- **Smart File Categorization**: Documents, videos, and archives
- **Smart Content Filtering**: Only includes user-facing files, excludes internal development files
- **Search Index Generation**: Creates optimized search index for client-side queries

### File Type Recognition (User-Facing Only)
- **Documents**: PDF, DOCX, PPTX, TXT
- **Media**: MP4, MP3
- **Archives**: ZIP

### Advanced Features
- **Size Warnings**: Flags files over 50MB for download warnings
- **Human-Readable Sizes**: Converts bytes to KB/MB/GB format
- **Performance Stats**: Tracks processing time and throughput
- **Error Handling**: Graceful handling of permission issues and corrupted files

## 📊 Generated Manifest Structure

```json
{
  "generated": "2024-01-15T10:30:00Z",
  "totalSize": 26843545600,
  "totalFiles": 1247,
  "totalFolders": 312,
  "sections": [
    {
      "id": "section-01",
      "name": "Section 01 - Materials for the Rapture Kit Sender",
      "path": "Section 01 - Materials for the Rapture Kit Sender/",
      "size": 5432109876,
      "fileCount": 156,
      "folderCount": 23,
      "folders": [...],
      "files": [
        {
          "name": "Tutorial 1 - How To Copy a Rapture Kit USB to Your PC or Mac.mp4",
          "path": "Section 01 - Materials for the Rapture Kit Sender/Tutorial 1 - How To Copy a Rapture Kit USB to Your PC or Mac.mp4",
          "size": 156789456,
          "type": "video/mp4",
          "category": "video",
          "icon": "🎥",
          "modified": "2024-01-05T12:00:00Z",
          "downloadable": true,
          "sizeWarning": true,
          "sizeHuman": "149.5 MB"
        }
      ]
    }
  ],
  "fileTypes": {
    "video": {
      "count": 45,
      "totalSize": 15000000000,
      "totalSizeHuman": "14.0 GB",
      "extensions": [".mp4"]
    },
    "document": {
      "count": 234,
      "totalSize": 850000000,
      "totalSizeHuman": "810.6 MB", 
      "extensions": [".pdf", ".docx"]
    }
  },
  "searchIndex": {
    "files": [...],
    "folders": [...],
    "keywords": [...]
  }
}
```

## ⚙️ Command Line Options

| Option | Short | Description | Default |
|--------|-------|-------------|---------|
| `--output` | `-o` | Output file path | `file-manifest.json` |
| `--path` | `-p` | Root directory to scan | Current directory |
| `--verbose` | `-v` | Enable detailed logging | `false` |
| `--help` | `-h` | Show help message | - |

## 🧪 Testing

Run the built-in test suite to verify functionality:

```bash
# Run comprehensive tests
npm test

# Or directly
node test-manifest.js
```

The test creates a mock file structure, generates a manifest, and validates:
- ✅ Correct JSON structure
- ✅ File type detection
- ✅ Content filtering rules
- ✅ Size calculations
- ✅ Search index generation

## 📁 Content Filtering

The generator automatically includes only user-facing content:
- **Included**: Section 02-09 (all user content)
- **Included**: Section 01 Tutorial MP4 files (Tutorial 1-5)
- **Included**: _Read Me First.pdf in root directory  
- **Excluded**: Section 01/webifyRK/* (internal website files)
- **File Types**: Only PDF, DOCX, PPTX, TXT, MP4, MP3, ZIP

This ensures the file browser shows only downloadable content relevant to users, including the important Tutorial videos.

## 🎯 Performance

Optimized for large datasets:
- **26GB content**: Processes efficiently with minimal memory usage
- **1200+ files**: Handles large file counts without slowdown  
- **300+ folders**: Recursive scanning with progress tracking
- **Concurrent processing**: Parallel file stat operations where possible

Example performance on typical content:
```
📊 Generation Statistics:
   📁 Sections processed: 9
   📂 Total folders: 312
   📄 Total files: 1,247
   💾 Total size: 25.0 GB
   ⏱️  Generation time: 3,456ms
   🚀 Files/second: 360
```

## 🔍 Integration with File Browser

The generated manifest integrates with the file browser system:

1. **Tree View**: Uses nested folder structure for navigation
2. **Search System**: Leverages pre-built search index
3. **Download Management**: File size warnings and categorization
4. **Performance**: Client-side operations without server calls

## 📝 Example Usage

### Basic Generation
```bash
# Generate manifest for current directory
node generate-manifest.js

# Output: file-manifest.json created
```

### Advanced Generation
```bash
# Scan specific directory with verbose output
node generate-manifest.js \
  --path "/path/to/content" \
  --output "custom-manifest.json" \
  --verbose

# Output shows detailed progress and statistics
```

### Programmatic Usage
```javascript
const ManifestGenerator = require('./generate-manifest.js');

const generator = new ManifestGenerator({
    path: './my-content',
    output: 'my-manifest.json',
    verbose: true
});

await generator.generate();
```

## 🛠️ Troubleshooting

### Common Issues

**Permission Errors**
```bash
# Ensure read access to all directories
chmod -R +r ./content
```

**Large File Timeouts**
- Generator handles large files efficiently
- Uses streaming for file stats to avoid memory issues

**Missing Files in Output**
- Ensure files have supported extensions (PDF, DOCX, PPTX, TXT, MP4, MP3, ZIP)
- Check if files are in webifyRK directory (excluded by design)
- Use `--verbose` flag to see processing details

### Debug Mode
```bash
# Enable verbose logging to see detailed processing
node generate-manifest.js --verbose

# Example output:
# 📂 Processing section: Section 02 - First things first
# 📂 Processing section: Section 03 - Bibles  
# 💾 Manifest written to: file-manifest.json
```

## 🔄 Updating the Manifest

Since your content is static and doesn't change frequently:

1. **Initial Generation**: Run once to create the complete manifest
2. **Periodic Updates**: Re-run only when content is added/changed
3. **Automation**: Could be integrated into a deployment pipeline if needed

The manifest generation is fast enough to run on-demand when needed.

## 📈 Next Steps

After generating your manifest:

1. **Implement File Browser**: Use manifest with the tree view and details components
2. **Add Search**: Leverage the generated search index for fast queries  
3. **Download Management**: Use file size warnings and categories
4. **Caching**: Consider caching the manifest client-side for performance

The manifest provides everything needed for a full-featured static file browser system!
